-------------------------------------------------------------------------------
--
-- Simple VGA raster display
--
-- Stephen A. Edwards
-- sedwards@cs.columbia.edu
--
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_arith.CONV_STD_LOGIC_VECTOR;

entity de2_vga_raster is
  
  port (
    reset : in std_logic;
    clk   : in std_logic;                    -- Should be 25.125 MHz

	read : in std_logic;
	write : in std_logic;
	chipselect : in std_logic;
	address : in unsigned(4 downto 0);
	readdata : out unsigned(15 downto 0);
	writedata : in unsigned(15 downto 0);

    VGA_CLK,                         -- Clock
    VGA_HS,                          -- H_SYNC
    VGA_VS,                          -- V_SYNC
    VGA_BLANK,                       -- BLANK
    VGA_SYNC : out std_logic;        -- SYNC
    VGA_R,                           -- Red[9:0]
    VGA_G,                           -- Green[9:0]
    VGA_B : out unsigned(9 downto 0) -- Blue[9:0]
    );

end de2_vga_raster;

architecture rtl of de2_vga_raster is
  
  -- Video parameters
  
  constant HTOTAL       : integer := 800;
  constant HSYNC        : integer := 96;
  constant HBACK_PORCH  : integer := 48;
  constant HACTIVE      : integer := 640;
  constant HFRONT_PORCH : integer := 16;
  
  constant VTOTAL       : integer := 525;
  constant VSYNC        : integer := 2;
  constant VBACK_PORCH  : integer := 33;
  constant VACTIVE      : integer := 480;
  constant VFRONT_PORCH : integer := 10;

  constant RECTANGLE_HSTART : integer := 100;
  constant RECTANGLE_HEND   : integer := 540;
  constant RECTANGLE_VSTART : integer := 100;
  constant RECTANGLE_VEND   : integer := 380;


  
  -- Signals for the video controller
  signal Hcount : unsigned(12 downto 0);  -- Horizontal position (0-800)
  signal Vcount : unsigned(12 downto 0);  -- Vertical position (0-524)
  signal EndOfLine, EndOfField : std_logic;

  signal vga_hblank, vga_hsync,
    vga_vblank, vga_vsync : std_logic;  -- Sync. signals


  
--Signals written to Avalon Bus
  
  --Paddle Y position
  signal paddleY: unsigned(12 downto 0) := "0000100100010";
  --X and Y coordinates at the center of the ball
  signal XCO : unsigned(12 downto 0) := "0000110010000";
  signal YCO : unsigned(12 downto 0) := "0000100100010";

  --Scores
  signal myScore: unsigned(3 downto 0) := "0000";
  signal compScore: unsigned(3 downto 0) := "0000";
  
  --Denote whether blocks have been destroyed
  signal compBlock1: std_logic := '1';
  signal compBlock2: std_logic := '1';
  signal compBlock3: std_logic := '1';
  signal compBlock4: std_logic := '1';
  signal compBlock5: std_logic := '1';
  signal compBlock6: std_logic := '1';

  --Signals to track charging and deployment of bullets
  signal bulletInFlight: std_logic;
  signal bulletCharged: unsigned(1 downto 0):= "00";
  signal bulletX: unsigned(12 downto 0):= "0000000000000";
  signal bulletY: unsigned(12 downto 0):= "0000000000000";

  --logic signals to determine how to paint a pixel
  signal paintRed : std_logic := '1';
  signal paintGreen: std_logic := '1';
  signal paintCompBlock: std_logic;
  signal paintBullet: std_logic;
  signal paintPaddle : std_logic;
  signal Cir : std_logic;


 --Used to determine if a pixel is inside the circle radius
 signal Xd : unsigned(12 downto 0);
  signal Yd : unsigned(12 downto 0);
  

--Constants
constant RADI : unsigned(12 downto 0) := "0000000001000";
constant PADDLEX: unsigned(12 downto 0) := "0000010100000";
constant BULLET_WIDTH: unsigned(12 downto 0) := "0000000001010";
constant BULLET_HEAD_LENGTH: unsigned(12 downto 0) := "0000000000101";

begin

  GetCoords : process (clk)
  begin
	if rising_edge(clk) then
	  if (write = '1') AND (chipselect = '1') then
		if address = "00001" then
			paddleY <= writedata(12 downto 0);
		elsif address = "00010" then
		  XCO <= writedata( 12 downto 0);
	    elsif address = "00000" then
		  YCO <= writedata( 12 downto 0);
		elsif address = "00011" then
		  myScore <= writedata( 3 downto 0);
	    elsif address = "00100" then
		  compScore <= writedata( 3 downto 0);
	    
		elsif address = "00101" then
		  compBlock1 <= writedata(0);
		elsif address = "00110" then
		  compBlock2 <= writedata(0);
		elsif address = "00111" then
		  compBlock3 <= writedata(0);
		elsif address = "01000" then
		  compBlock4 <= writedata(0);
		elsif address = "01001" then
		  compBlock5 <= writedata(0);
		elsif address = "01010" then
		  compBlock6 <= writedata(0);
		
		elsif address = "01011" then
		  bulletInFlight <= writedata(0);
		elsif address = "01100" then
		  bulletCharged <= writedata(1 downto 0);
		elsif address = "01101" then
		  bulletX <= writedata(12 downto 0);
		elsif address = "01110" then
		  bulletY <= writedata(12 downto 0);
		
		
	    end if;
	  end if;
    end if;
  end process GetCoords;

  SpitCoords : process (clk)
  begin 
	if rising_edge(clk) then
	  if (read = '1') and (chipselect = '1') then
		if address = "00001" then
		  readdata(15 downto 13) <= "000";
		  readdata(12 downto 0) <=  paddleY ;
		elsif address = "00010" then
		  readdata(15 downto 13) <= "000";
		  readdata(12 downto 0) <=  XCO ;
	    elsif address = "00000" then
		  readdata(15 downto 13) <= "000";
		  readdata(12 downto 0) <=  YCO ;
		else
			readdata(15 downto 0) <= "0000000000000000";
	    end if;
	  end if;
   end if;
  end process SpitCoords;

  -- Horizontal and vertical counters

  HCounter : process (clk)
  begin
    if rising_edge(clk) then      
      if reset = '1' then
        Hcount <= (others => '0');
      elsif EndOfLine = '1' then
        Hcount <= (others => '0');
      else
        Hcount <= Hcount + 1;
      end if;      
    end if;
  end process HCounter;

  EndOfLine <= '1' when Hcount = HTOTAL - 1 else '0';
  
  VCounter: process (clk)
  begin
    if rising_edge(clk) then      
      if reset = '1' then
        Vcount <= (others => '0');
      elsif EndOfLine = '1' then
        if EndOfField = '1' then
          Vcount <= (others => '0');
        else
          Vcount <= Vcount + 1;
        end if;
      end if;
    end if;
  end process VCounter;

  EndOfField <= '1' when Vcount(9 downto 0) = VTOTAL - 1 else '0';

  -- State machines to generate HSYNC, VSYNC, HBLANK, and VBLANK

  HSyncGen : process (clk)
  begin
    if rising_edge(clk) then     
      if reset = '1' or EndOfLine = '1' then
        vga_hsync <= '1';
      elsif Hcount(9 downto 0) = HSYNC - 1 then
        vga_hsync <= '0';
      end if;
    end if;
  end process HSyncGen;
  
  HBlankGen : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        vga_hblank <= '1';
      elsif Hcount(9 downto 0) = HSYNC + HBACK_PORCH then
        vga_hblank <= '0';
      elsif Hcount(9 downto 0) = HSYNC + HBACK_PORCH + HACTIVE then
        vga_hblank <= '1';
      end if;      
    end if;
  end process HBlankGen;

  VSyncGen : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        vga_vsync <= '1';
      elsif EndOfLine ='1' then
        if EndOfField = '1' then
          vga_vsync <= '1';
        elsif Vcount(9 downto 0) = VSYNC - 1 then
          vga_vsync <= '0';
        end if;
      end if;      
    end if;
  end process VSyncGen;

  VBlankGen : process (clk)
  begin
    if rising_edge(clk) then    
      if reset = '1' then
        vga_vblank <= '1';
      elsif EndOfLine = '1' then
        if Vcount(9 downto 0) = VSYNC + VBACK_PORCH - 1 then
          vga_vblank <= '0';
        elsif Vcount(9 downto 0) = VSYNC + VBACK_PORCH + VACTIVE - 1 then
          vga_vblank <= '1';
        end if;
      end if;
    end if;
  end process VBlankGen;


 

  VideoOut: process (clk, reset)
  begin


	  --Determine whether to paint ball
	  if ( Hcount > XCO ) then
		Xd <= Hcount - XCO;
  	  else
		Xd <= XCO - Hcount;
	  end if;
	  if ( Vcount > YCO ) then
		Yd <= Vcount - YCO;
	  else
		Yd <= YCO - Vcount;
	  end if;	
	
	  if ( Xd <= RADI ) and ( Yd <= RADI ) then
	    if (reset = '1') or ( Xd * Xd + Yd * Yd > RADI * RADI ) then
		  Cir <= '0';
	    elsif ( Xd * Xd + Yd * Yd < RADI * RADI) then
		  Cir <= '1';
        end if;
      else 
        Cir <= '0';
	  end if;
	
	
	--Determine whether to paint paddle
	if (Vcount <= paddleY + 100) and (Vcount >= paddleY) then
		if (Hcount >= PADDLEX) and (Hcount <= PADDLEX + 10) then
			paintPaddle <= '1';
		else
			paintPaddle <= '0';
		end if;
	else
		paintPaddle <= '0';
	end if;
	
	--Determine whether to paint a block
	if(HCount >= 750) and (HCount < 800) then
		if(Vcount > 38) and (Vcount <=116) and (compBlock1 = '1') then
			paintCompBlock <= '1';
		elsif(Vcount > 117) and (Vcount <=195) and (compBlock2 = '1') then
			paintCompBlock <= '1';
		elsif(Vcount > 196) and (Vcount <=274) and (compBlock3 = '1') then
			paintCompBlock <= '1';
		elsif(Vcount > 275) and (Vcount <=353) and (compBlock4 = '1') then
			paintCompBlock <= '1';
		elsif(Vcount > 354) and (Vcount <=432) and (compBlock5 = '1') then
			paintCompBlock <= '1';
		elsif(Vcount > 433) and (Vcount <=511) and (compBlock6 = '1') then
			paintCompBlock <= '1';
		else
			paintCompBlock <='0';
		end if;
	else
		paintCompBlock <= '0';
	end if;

	--Determine whether to paint a score square (green if score reached)
	if (Vcount <= 50) and (Vcount > 40) then
		if(Hcount >= 200) and (Hcount <210) then
			if (myScore <= 0) then
				paintRed <= '1';
				paintGreen <= '0';
			else
				paintRed <= '0';
				paintGreen <= '1';
			end if;
		
		elsif(Hcount >= 220) and (Hcount <230) then
			if (myScore <= 1) then
				paintRed <= '1';
				paintGreen <= '0';
			else
				paintRed <= '0';
				paintGreen <= '1';
			end if;
			
		elsif(Hcount >= 240) and (Hcount <250) then
			if (myScore <= 2) then
				paintRed <= '1';
				paintGreen <= '0';
			else
				paintRed <= '0';
				paintGreen <= '1';
			end if;
		
		elsif(Hcount >= 260) and (Hcount <270) then
			if (myScore <= 3) then
				paintRed <= '1';
				paintGreen <= '0';
			else
				paintRed <= '0';
				paintGreen <= '1';
			end if;
		
		elsif(Hcount >= 280) and (Hcount <290) then
			if (myScore <= 4) then
				paintRed <= '1';
				paintGreen <= '0';
			else
				paintRed <= '0';
				paintGreen <= '1';
			end if;
		
		
		
		elsif(Hcount >= 600) and (Hcount <610) then
			if (compScore <= 0) then
				paintRed <= '1';
				paintGreen <= '0';
			else
				paintRed <= '0';
				paintGreen <= '1';
			end if;
		
		elsif(Hcount >= 620) and (Hcount <630) then
			if (compScore <= 1) then
				paintRed <= '1';
				paintGreen <= '0';
			else
				paintRed <= '0';
				paintGreen <= '1';
			end if;
		
		elsif(Hcount >= 640) and (Hcount <650) then
			if (compScore <= 2) then
				paintRed <= '1';
				paintGreen <= '0';
			else
				paintRed <= '0';
				paintGreen <= '1';
			end if;
		
		elsif(Hcount >= 660) and (Hcount <670) then
			if (compScore <= 3) then
				paintRed <= '1';
				paintGreen <= '0';
			else
				paintRed <= '0';
				paintGreen <= '1';
			end if;
		
		elsif(Hcount >= 680) and (Hcount <690) then
			if (compScore <= 4) then
				paintRed <= '1';
				paintGreen <= '0';
			else
				paintRed <= '0';
				paintGreen <= '1';
			end if;
		
		else
			paintRed <= '0';
			paintGreen <= '0';
		end if;
	else
		paintRed <= '0';
		paintGreen <= '0';
	end if;
			
			
	--Determines whether to paint a bullet (charging or in flight)
	if (bulletInFlight = '1') or (bulletCharged > 0) then
		if (Vcount >= bulletY) and (Vcount < bulletY + BULLET_WIDTH) then
			if (Hcount >=bulletX) and (HCount < bulletX + BULLET_HEAD_LENGTH) then
				paintBullet <= '1';
			elsif (Hcount >=bulletX) and (HCount < bulletX + BULLET_WIDTH) then
				if (bulletCharged > 1) or (bulletInFlight = '1') then
					paintBullet <= '1';
				else
					paintBullet <= '0';
				end if;
			elsif (bulletCharged = "11") or bulletInFlight = '1' then
				if (Hcount >=bulletX) and (HCount < bulletX + BULLET_WIDTH+BULLET_HEAD_LENGTH) then
					if(Vcount - bulletY >= HCount - bulletX - BULLET_WIDTH) and (Vcount - bulletY <= BULLET_HEAD_LENGTH) then
						paintBullet <= '1';
					elsif(bulletY + BULLET_WIDTH - Vcount >= HCount - bulletX - BULLET_WIDTH) and (Vcount - bulletY >= BULLET_HEAD_LENGTH) then
						paintBullet <= '1';
					else
						paintBullet <= '0';
					end if;
				else
					paintBullet <= '0';
				end if;
			else
				paintBullet <= '0';
			end if;
		else
			paintBullet <= '0';
		end if;
	else
		paintBullet <= '0';	
	end if;
	
	--Choose color palette based on logic signals
    if reset = '1' then
      VGA_R <= "0000000000";
      VGA_G <= "0000000000";
      VGA_B <= "0000000000";
    elsif clk'event and clk = '1' then
      if Cir = '1' then
        VGA_R <= "1111111111";
        VGA_G <= "1111111111";
        VGA_B <= "1111111111";
	  elsif paintBullet = '1' then
        VGA_R <= "1111111111";
        VGA_G <= "0010011011";
        VGA_B <= "0000000000";  
	  elsif paintPaddle = '1' then
		VGA_R <= "1010101010";
        VGA_G <= "1010101010";
        VGA_B <= "1010101010";
	 elsif paintRed = '1' then
        VGA_R <= "1111111111";
        VGA_G <= "0000000000";
        VGA_B <= "0000000000";
      elsif paintGreen = '1' then
        VGA_R <= "0000000000";
        VGA_G <= "1111111111";
        VGA_B <= "0000000000";
	  elsif paintCompBlock = '1' then
        VGA_R <= "1010101010";
        VGA_G <= "1010101010";
        VGA_B <= "1010101010";  
     	
      elsif vga_hblank = '0' and vga_vblank ='0' then
        VGA_R <= "0000000000";
        VGA_G <= "0000000000";
        VGA_B <= "0000000000";
    
	   else
        VGA_R <= "0000000000";
        VGA_G <= "0000000000";
        VGA_B <= "0000000000";    
      end if;
    end if;
  end process VideoOut;

  VGA_CLK <= clk;
  VGA_HS <= not vga_hsync;
  VGA_VS <= not vga_vsync;
  VGA_SYNC <= '0';
  VGA_BLANK <= not (vga_hsync or vga_vsync);

end rtl;
